import { Event } from './Modules/Event';

import { FullLeaderboard } from "./FullLeaderboard";
import { DestructionHelper } from './DestructionHelper';
import {
    findParentComponent,
    findSoWithComponent,
    getOrCreateScreenTransform,
    setRenderLayerRecursively,
    setRenderOrderRecursivelyRelativeToParent
} from './ComponentUtils';
import {
    BackgroundCustomization,
    IDependencies,
    LeaderboardDetails,
    LeaderboardInitializationOptions,
    LeaderboardParams,
    LeaderboardRecordsWrapper,
    ScoreResetIntervalOption
} from "./Interfaces/LeaderboardRelated";
import { CameraBadgeLeaderboard, } from "./CameraBadgeLeaderboard";
import { LeaderboardConstants, } from "./Interfaces/LeaderboardConstants";
import { isSameUserEntry } from "./LeaderboardUtils";
import { ComponentWithDebug } from "./Modules/ComponentWithDebug";
import { BitmojiStickerLoader } from "./BitmojiStickerLoader";
import { CachedEvent } from "./Modules/CachedEvent";
import {
    constructLeaderboardId,
    getIdFromUTCTime,
    getMilisecondsToFinish,
    getUTCTime,
    isDateStringValid
} from "./Modules/DateUtils";
import { TokenManager } from "./TokenManager";
import UserRecord = Leaderboard.UserRecord;
import UsersType = Leaderboard.UsersType;

enum ScoreLifetimeSeconds {
    Day = 86400,
    Week = 604800,
    Month = 2592000,
    Year = 31104000
}

const ScoreLifetimeOptionsToScoreLifetimeSeconds = {
    [ScoreResetIntervalOption.Day]:  ScoreLifetimeSeconds.Day,
    [ScoreResetIntervalOption.Week]: ScoreLifetimeSeconds.Week,
    [ScoreResetIntervalOption.Month]: ScoreLifetimeSeconds.Month,
    [ScoreResetIntervalOption.Year]: ScoreLifetimeSeconds.Year,
};

const ScoreLifetimeOptionsToResetString = {
    [ScoreResetIntervalOption.Day]:  LeaderboardConstants.FullLeaderboard.DAILY_RESET_STRING,
    [ScoreResetIntervalOption.Week]: LeaderboardConstants.FullLeaderboard.WEEKLY_RESET_STRING,
    [ScoreResetIntervalOption.Month]: LeaderboardConstants.FullLeaderboard.MONTHLY_RESET_STRING,
    [ScoreResetIntervalOption.Year]: LeaderboardConstants.FullLeaderboard.YEARLY_RESET_STRING,
};

@component
export class LeaderboardComponent extends ComponentWithDebug {
    @input('string', '')
    @hint('Changing leaderboard name will reset scores')
    private readonly leaderboardName: string;

    @input('boolean', 'true')
    private readonly autoInitialize: boolean;

    @input('int', '1')
    @label("Score Reset Interval")
    @widget(new ComboBoxWidget()
        .addItem('Day', 0)
        .addItem('Week', 1)
        .addItem('Month', 2)
        .addItem('Year', 3))
    private readonly scoreLifetimeOption: ScoreResetIntervalOption;

    @input('string', '12/31/2024')
    @hint('In format mm/dd/yyyy')
    private readonly leaderboardStartDate: string;

    @input('int', '1')
    @widget(new ComboBoxWidget()
        .addItem('Descending', 1)
        .addItem('Ascending', 2))
    private readonly scoreOrdering: Leaderboard.OrderingType;

    @input('int', '10')
    @hint('Maximum user limit is 50')
    private userLimit: number;
    @ui.separator

    @input('int', '1')
    private readonly renderOrder: number = 1;

    @input('int', '2')
    @widget(new ComboBoxWidget()
        .addItem('None', 0)
        .addItem('Bitmoji', 1)
        .addItem('Texture', 2))
    private readonly backgroundCustomization: BackgroundCustomization;

    @input()
    @showIf('backgroundCustomization', 2)
    private readonly backgroundTexture: Texture;

    @input()
    @showIf('backgroundCustomization', 1)
    @hint('Please enter bitmoji sticker id or leave empty to use selfie sticker')
    private customStickerId: string;

    @ui.group_start('Customisable Prefabs')
    @showIf('customisablePrefabs')

    @input('Asset.ObjectPrefab')
    @hint('This Prefab contains LeaderboardEntry script which can be customised')
    private readonly leaderboardEntry: ObjectPrefab;

    @input('Asset.ObjectPrefab')
    private readonly cameraBadgePrefab: ObjectPrefab;
    @ui.group_end
    @ui.separator

    @input('Asset.ObjectPrefab')
    private readonly leaderboardVisuals: ObjectPrefab;

    @input('Asset.ObjectPrefab')
    private readonly leaderboardEntryFallback: ObjectPrefab;

    @input('Asset.ObjectPrefab')
    private readonly componentDependencies: ObjectPrefab;

    @input()
    private readonly useCameraBadge: boolean = false;

    @input()
    private readonly leaderboardModule: LeaderboardModule;

    @input()
    private readonly bitmojiModule: BitmojiModule;

    @input()
    private readonly remoteMediaModule: RemoteMediaModule;

    public ScoreResetIntervalOption = ScoreResetIntervalOption;

    onLeaderboardRecordsUpdated: CachedEvent<LeaderboardRecordsWrapper> = new CachedEvent();
    onShow: Event = new Event();
    onHide: Event = new Event();

    private readonly userType: Leaderboard.UsersType = Leaderboard.UsersType.Friends;
    private readonly destructionHelper = new DestructionHelper();
    protected readonly screenTransform = getOrCreateScreenTransform(this.getSceneObject(), this.destructionHelper);
    private isVisible: boolean = false;
    private scoreLifetimeSeconds: number;
    private currentUserResolver: Function;
    private currentUserBitmojiResolver: Function;
    private getLeaderboardResolver: Function;
    private getLeaderboardRejecter: Function;
    private currentUserPromise: Promise<SnapchatUser>;
    private currentUserBitmojiPromise: Promise<Texture>;
    private getLeaderboardPromise: Promise<Leaderboard>;
    private leaderboard: Leaderboard = null;
    private dependencies: IDependencies = null;
    private userBitmojiTexture: Texture = null;
    private cameraBadgeLeaderboard: CameraBadgeLeaderboard;
    private leaderboardVisualsControl: FullLeaderboard;
    private currentUserRecord: Leaderboard.UserRecord = null;
    private allMergedRecords: Leaderboard.UserRecord[] = [];
    private bitmojiStickerLoader: BitmojiStickerLoader;
    private useTimer: boolean = true;
    private leaderboardParams: LeaderboardParams = null;
    private updateEvent: UpdateEvent = null;
    private tokenManager: TokenManager = new TokenManager();
    private currentToken: number;

    onAwake() {
        this.createEvent("OnDestroyEvent").bind(() => {
            this.destructionHelper.destroyAll();
        });

        this.updateEvent = this.createEvent("UpdateEvent");
        this.updateEvent.enabled = false;
        this.updateEvent.bind(this.onUpdate);

        if (!this.isInputValid()) {
            return;
        }

        this.bitmojiStickerLoader = new BitmojiStickerLoader(this.bitmojiModule, this.remoteMediaModule);

        const dependenciesParent: SceneObject = this.componentDependencies.instantiate(this.getSceneObject().getParent());
        dependenciesParent.enabled = true;

        if (!this.isTweenJsPresent()) {
            return;
        }

        this.currentUserPromise = new Promise(<SnapchatUser> (resolve, reject) => {
            this.currentUserResolver = resolve;
        });

        this.currentUserBitmojiPromise = new Promise<Texture> ((resolve, reject) => {
            this.currentUserBitmojiResolver = resolve;
        });

        this.getLeaderboardPromise = new Promise(<SnapchatUser> (resolve, reject) => {
            this.getLeaderboardResolver = resolve;
            this.getLeaderboardRejecter = reject;
        });

        this.dependencies = dependenciesParent.getComponent("Component.ScriptComponent") as IDependencies;
        this.dependencies.enabled = true;

        this.initializeSideSwitcher();
        this.adjustUserLimit();
        this.initializeLeaderboardVisuals();

        setRenderOrderRecursivelyRelativeToParent(this.getSceneObject(), this.renderOrder);
        setRenderLayerRecursively(this.getSceneObject(), this.getSceneObject().layer);

        this.scoreLifetimeSeconds = ScoreLifetimeOptionsToScoreLifetimeSeconds[this.scoreLifetimeOption];
        this.leaderboardVisualsControl.setFallbackEnabled(true);

        if (this.backgroundCustomization === BackgroundCustomization.Texture) {
            this.leaderboardVisualsControl.setCustomTextureHeader(this.backgroundTexture);
        }

        if (this.autoInitialize) {
            this.leaderboardVisualsControl.setTimerUiEnabled(true);
            this.createLeaderboard(this.collectParamsFromInput());
            this.leaderboardVisualsControl.setTimerText(ScoreLifetimeOptionsToResetString[this.scoreLifetimeOption]);
        }
    }

    get visible(): boolean {
        return this.isVisible;
    }

    public submitScore(score: number): void {
        this.printDebug("Submitting score " + score);
        if (!isNull(this.leaderboard)) {
            this.leaderboard.submitScore(score, this.onSubmitScoreSuccess, this.onSubmitScoreFailed);
        }
    }

    public show(): void {
        this.leaderboardVisualsControl.show();
    }

    public hide(): void {
        this.leaderboardVisualsControl.hide();
    }

    public getSideSwitcher(): SideSwitcher {
        return this.dependencies.getSideSwitcher();
    }

    public async getLeaderboard(): Promise<Leaderboard> {
        return this.getLeaderboardPromise;
    }

    public async getCurrentUser(): Promise<SnapchatUser> {
        return this.currentUserPromise;
    }

    public async getCurrentUserBitmoji(): Promise<Texture> {
        return this.currentUserBitmojiPromise;
    }

    public initializeWithOptions(leaderboardInitializationOptions: LeaderboardInitializationOptions): void {
        if (this.autoInitialize) {
            throw new Error("Please remove Auto Initialize from component inputs");
        }

        if (isNull(leaderboardInitializationOptions.userType) ||
            isNull(leaderboardInitializationOptions.scoreOrdering) ||
            isNull(leaderboardInitializationOptions.userLimit) ||
            isNull(leaderboardInitializationOptions.scoreResetInterval) ||
            isNull(leaderboardInitializationOptions.useTimer) ||
            isNull(leaderboardInitializationOptions.name)
        ) {
            this.printWarning("LeaderboardInitializationOptions are incorrect");
            throw new Error("LeaderboardInitializationOptions are incorrect");
        }

        const isNameProvided = !isNull(leaderboardInitializationOptions.name);
        if (!leaderboardInitializationOptions.useTimer && !isNameProvided) {
            throw new Error("Please provide leaderboard name");
        }

        if (leaderboardInitializationOptions.useTimer && isNull(leaderboardInitializationOptions.leaderboardStartDate)) {
            throw new Error("Please provide leaderboard start date");
        }

        const ttlSeconds = ScoreLifetimeOptionsToScoreLifetimeSeconds[leaderboardInitializationOptions.scoreResetInterval];

        this.useTimer = leaderboardInitializationOptions.useTimer;
        if (this.useTimer) {
            this.leaderboardVisualsControl.setTimerText(ScoreLifetimeOptionsToResetString[leaderboardInitializationOptions.scoreResetInterval]);
        }

        const scoreOrdering = leaderboardInitializationOptions.scoreOrdering;
        const userType = leaderboardInitializationOptions.userType;
        const userLimit = !isNull(leaderboardInitializationOptions.userLimit) ? leaderboardInitializationOptions.userLimit : this.userLimit;

        const details = this.createNewLeaderboardDetails(leaderboardInitializationOptions.leaderboardStartDate,
            leaderboardInitializationOptions.name,
            ttlSeconds,
            leaderboardInitializationOptions.scoreResetInterval,
            scoreOrdering);
        const name = isNameProvided ? leaderboardInitializationOptions.name : details.leaderboardId;

        const leaderboardParams = {
            leaderboardName: name,
            ttlSeconds: ttlSeconds,
            scoreLifetimeOption: leaderboardInitializationOptions.scoreResetInterval,
            orderingType: scoreOrdering,
            userType: userType,
            userLimit: userLimit,
            leaderboardDetails: leaderboardInitializationOptions.useTimer ? details : null,
            leaderboardNamePrefix: leaderboardInitializationOptions.name
        };

        this.createLeaderboard(leaderboardParams);
    }

    public setLeaderboardName(name: string) {
        if (!(this.leaderboardVisualsControl instanceof CameraBadgeLeaderboard)) {
            this.leaderboardVisualsControl.setLeaderboardName(name);
        }
    }

    private createNewLeaderboardDetails(leaderboardStartDateString: string, leaderboardNamePrefix: string,
        scoreLifetimeSeconds: number, scoreLifetimeOption: ScoreResetIntervalOption, scoreOrdering: Leaderboard.OrderingType): LeaderboardDetails {

        const currentDate = new Date();
        const scoreLifetimeMillis = scoreLifetimeSeconds * 1000;

        const parsedStartDate = new Date(Date.parse(leaderboardStartDateString));

        const offsetDateMillis = (Date.UTC(parsedStartDate.getFullYear(), parsedStartDate.getMonth(), parsedStartDate.getDate(), 0, 0, 0, 0));

        const leaderboardRelativeIndex = getIdFromUTCTime(getUTCTime(currentDate), scoreLifetimeMillis, -offsetDateMillis);

        const leaderboardDetails = {
            currentDate: currentDate,
            offsetDateMillis: offsetDateMillis,
            leaderboardRelativeIndex: leaderboardRelativeIndex,
            leaderboardId: constructLeaderboardId(leaderboardNamePrefix,
                leaderboardStartDateString,
                leaderboardRelativeIndex.toString(),
                scoreLifetimeOption.toString(),
                scoreOrdering.toString()),
            timeToFinish: getMilisecondsToFinish(offsetDateMillis,
                scoreLifetimeMillis,
                leaderboardRelativeIndex,
                getUTCTime(currentDate)),
            leaderboardStartDate: leaderboardStartDateString,
        };
        return leaderboardDetails;
    }

    private updateLeaderboardDetails(): void {
        const currentDate = new Date();
        if (isNull(this.leaderboardParams.leaderboardDetails)) {
            return;
        }

        const timeToFinishMs = getMilisecondsToFinish(this.leaderboardParams.leaderboardDetails.offsetDateMillis,
            this.leaderboardParams.ttlSeconds * 1000,
            this.leaderboardParams.leaderboardDetails.leaderboardRelativeIndex,
            getUTCTime(currentDate));

        if (timeToFinishMs <= 0) {
            this.reset();
            this.recreateLeaderboard();
            return;
        }

        this.leaderboardParams.leaderboardDetails.timeToFinish = timeToFinishMs;
        this.leaderboardVisualsControl.setTimeLeft(timeToFinishMs);
    }

    private collectParamsFromInput(): LeaderboardParams {
        const leaderboardDetails = this.createNewLeaderboardDetails(this.leaderboardStartDate,
            this.leaderboardName,
            this.scoreLifetimeSeconds,
            this.scoreLifetimeOption,
            this.scoreOrdering);

        return {
            leaderboardName: leaderboardDetails.leaderboardId,
            ttlSeconds: this.scoreLifetimeSeconds,
            scoreLifetimeOption: this.scoreLifetimeOption,
            orderingType: this.scoreOrdering,
            userType: this.userType,
            userLimit: this.userLimit,
            leaderboardDetails: leaderboardDetails,
            leaderboardNamePrefix: this.leaderboardName
        };
    }

    private isTweenJsPresent(): boolean {
        if (!global.TWEEN || !global.TWEEN.Tween) {
            this.printWarning("Please add Tween Manager");
            return false;
        }
        return true;
    }

    private initializeSideSwitcher(): void {
        const sideSwitcher: SideSwitcher = this.dependencies.getSideSwitcher();
        sideSwitcher.enableInteractable();

        sideSwitcher.onSwitch.add(() => {
            this.leaderboardVisualsControl.show();
            sideSwitcher.hide();
        });

        sideSwitcher.onHide.add(() => {
            if (sideSwitcher.tooltip) {
                sideSwitcher.tooltip.hide();
            }
        });

        setRenderOrderRecursivelyRelativeToParent(sideSwitcher.getSceneObject(), this.renderOrder + 1);
        setRenderLayerRecursively(sideSwitcher.getSceneObject(), this.getSceneObject().layer);
        this.applySideSwitcherDepthFix(sideSwitcher.getSceneObject());
    }

    private applySideSwitcherDepthFix(sideSwitcherSo: SceneObject): void {
        const interactions = findSoWithComponent(sideSwitcherSo, "Component.InteractionComponent");
        interactions.forEach((soWithInteraction) => {
            const screenTransform = soWithInteraction.getComponent("Component.ScreenTransform");
            if (!isNull(screenTransform)) {
                screenTransform.position = new vec3(screenTransform.position.x, screenTransform.position.y, 0);
            }
        });
    }

    private initializeLeaderboardVisuals(): void {
        if (this.useCameraBadge) {
            this.initializeCameraBadge();
        }

        this.leaderboardVisualsControl = new FullLeaderboard(this, this.leaderboardVisuals, this.leaderboardEntry, this.leaderboardEntryFallback, () => this.onLeaderboardClosed(), () => this.onLeaderboardOpened(), this.renderOrder, this.bitmojiStickerLoader, this.backgroundCustomization) as FullLeaderboard;
    }

    private adjustUserLimit(): void {
        if (this.userType === UsersType.Friends) {
            if (this.userLimit > LeaderboardConstants.Component.USER_LIMIT_FRIENDS) {
                this.printWarning("User limit was set to max value = " + LeaderboardConstants.Component.USER_LIMIT_FRIENDS);
            }
            this.userLimit = MathUtils.clamp(this.userLimit, 0, LeaderboardConstants.Component.USER_LIMIT_FRIENDS);
        } else if (this.userType === UsersType.Global) {
            if (this.userLimit > LeaderboardConstants.Component.USER_LIMIT_GLOBAL) {
                this.printWarning("User limit was set to max value = " + LeaderboardConstants.Component.USER_LIMIT_GLOBAL);
            }
            this.userLimit = MathUtils.clamp(this.userLimit, 0, LeaderboardConstants.Component.USER_LIMIT_GLOBAL);
        }
    }

    private onLeaderboardClosed(): void {
        this.dependencies.getSideSwitcher().show();
        this.dependencies.getSideSwitcher().enableInteractable();
        this.isVisible = false;
        this.onHide.trigger();
    }

    private onLeaderboardOpened(): void {
        this.isVisible = true;
        this.onShow.trigger();
    }

    private createLeaderboard(leaderboardParameters: LeaderboardParams): void {
        if (this.leaderboardParams !== null && this.leaderboardParams.token) {
            this.tokenManager.dismissToken(this.leaderboardParams.token);
        }
        const newToken = this.tokenManager.generateToken();
        leaderboardParameters.token = newToken;

        this.leaderboardParams = leaderboardParameters;
        this.leaderboardVisualsControl.setIsGlobalLeaderboard(leaderboardParameters.userType === UsersType.Global);

        const leaderboardCreateOptions = Leaderboard.CreateOptions.create();
        leaderboardCreateOptions.name = leaderboardParameters.leaderboardName;
        leaderboardCreateOptions.ttlSeconds = leaderboardParameters.ttlSeconds;
        leaderboardCreateOptions.orderingType = leaderboardParameters.orderingType;

        this.leaderboardModule.getLeaderboard(leaderboardCreateOptions, (leaderboard: Leaderboard) => {
            if (!this.tokenManager.isTokenValid(newToken)) {
                return;
            }
            this.printDebug("Leaderboard retrieved!");
            const retrievalOptions = this.createLeaderboardRetrievalOption(leaderboardParameters);
            this.leaderboard = leaderboard;
            this.getLeaderboardResolver(leaderboard);
            this.leaderboard.getLeaderboardInfo(retrievalOptions, (p1, p2) => this.onLeaderboardInfoRetrievedSuccesfull(p1, p2, newToken), this.onLeaderboardInfoRetrievedFailed);
        }, () => {
            this.getLeaderboardRejecter();
            this.printWarning("Leaderboard retrieval failed");
            this.leaderboardVisualsControl.setFallbackEnabled(true);
        });

        this.updateEvent.enabled = true;
    }

    private addEntriesIfDoesntExist(leaderboardRecords: Leaderboard.UserRecord[], currentUserRecord: Leaderboard.UserRecord): void {
        for (let i = 0; i < leaderboardRecords.length; i++) {
            let isEntryKnown = false;

            for (let j = 0; j < this.allMergedRecords.length; j++) {
                if (isSameUserEntry(this.allMergedRecords[j], leaderboardRecords[i])) {
                    this.allMergedRecords[j] = leaderboardRecords[i];
                    isEntryKnown = true;
                    break;
                }
            }
            if (!isEntryKnown) {
                this.allMergedRecords.push(leaderboardRecords[i]);
            }
        }

        if (!isNull(currentUserRecord)) {
            let isCurrentUserInRecords = false;
            for (let i = 0; i < this.allMergedRecords.length; i++) {
                if (isSameUserEntry(this.allMergedRecords[i], currentUserRecord)) {
                    isCurrentUserInRecords = true;
                    this.allMergedRecords[i] = currentUserRecord;
                    break;
                }
            }

            if (!isCurrentUserInRecords) {
                this.allMergedRecords.push(currentUserRecord);
            }
        }
    }

    private sortByScore(leaderboardRecords: Leaderboard.UserRecord[], orderingType: Leaderboard.OrderingType) {
        const compareFn = orderingType === Leaderboard.OrderingType.Descending ?
            (a, b) => b.score - a.score : (a, b) => a.score - b.score;
        leaderboardRecords.sort(compareFn);

        if (this.leaderboardParams.userType === UsersType.Global) {
            const compareFnGlobalRank =  (a, b) => a.globalExactRank - b.globalExactRank;
            leaderboardRecords.sort(compareFnGlobalRank);
        }
    }

    private initializeCameraBadge(): void {
        this.cameraBadgeLeaderboard = new CameraBadgeLeaderboard(this, this.cameraBadgePrefab, this.renderOrder);

        this.dependencies.getSideSwitcher().onShow.add(() => {
            this.cameraBadgeLeaderboard.show();
        });

        this.dependencies.getSideSwitcher().onHide.add(() => {
            this.cameraBadgeLeaderboard.hide();
        });
    }

    private loadBitmojiIfNotLoaded(): void {
        if (!isNull(this.userBitmojiTexture)) {
            return;
        }
        if (isNull(this.currentUserRecord) || isNull(this.currentUserRecord.snapchatUser)) {
            this.printDebug("Cannot load bitmoji");
            return;
        }

        const currentUser = this.currentUserRecord.snapchatUser;
        const hasBitmoji = !isNull(this.currentUserRecord.snapchatUser) &&
            !isNull(this.currentUserRecord.snapchatUser.hasBitmoji) && this.currentUserRecord.snapchatUser.hasBitmoji;

        if (!hasBitmoji) {
            return;
        }

        const onTextureReceived = (texture) => {
            this.currentUserBitmojiResolver(texture);
            this.userBitmojiTexture = texture;

            if (this.backgroundCustomization === BackgroundCustomization.Bitmoji && hasBitmoji) {
                this.leaderboardVisualsControl.setBitmoji(texture);
            }

            if (this.useCameraBadge && hasBitmoji) {
                this.cameraBadgeLeaderboard.setBitmoji(texture);
            }
        };

        const stickerId = (this.customStickerId && this.backgroundCustomization === BackgroundCustomization.Bitmoji)
            ? this.customStickerId : '';
        this.bitmojiStickerLoader.loadForUser(currentUser, stickerId)
            .then(onTextureReceived)
            .catch((e) => {
                this.printDebug("Error loading bitmoji");
            });
    }

    private onSubmitScoreSuccess = (currentUserRecord: UserRecord) => {
        this.printDebug("onSubmitScoreSuccess");
        this.onCurrentUserRecordUpdated(currentUserRecord);
        this.updateScoresState();
    };

    private onSubmitScoreFailed = (status: number) => {
        this.printDebug("onSubmitScoreFailed");
    };

    private onCurrentUserRecordUpdated(currentUserRecord: Leaderboard.UserRecord): void {
        if (isNull(currentUserRecord)) {
            return;
        }

        this.currentUserRecord = currentUserRecord;
        this.loadBitmojiIfNotLoaded();

        this.addEntriesIfDoesntExist([], currentUserRecord);
        this.sortByScore(this.allMergedRecords, this.leaderboardParams.orderingType);
    }

    private updateScoresState(): void {
        this.onLeaderboardRecordsUpdated.trigger({
            userRecords: this.allMergedRecords, currentUserRecord: this.currentUserRecord } as LeaderboardRecordsWrapper);

        if (!isNull(this.currentUserRecord) && !isNull(this.currentUserRecord.snapchatUser)) {
            this.currentUserResolver(this.currentUserRecord.snapchatUser);
        }

        this.leaderboardVisualsControl.visualiseEntries(this.allMergedRecords, this.currentUserRecord);

        if (this.useCameraBadge) {
            this.cameraBadgeLeaderboard.visualiseEntries(this.allMergedRecords, this.currentUserRecord);
        }
    }

    private onLeaderboardInfoRetrievedSuccesfull = (otherRecords: Leaderboard.UserRecord[], currentUserRecord: Leaderboard.UserRecord, token: number) => {
        if (!this.tokenManager.isTokenValid(token)) {
            return;
        }

        this.onCurrentUserRecordUpdated(currentUserRecord);
        this.addEntriesIfDoesntExist(otherRecords, currentUserRecord);
        this.sortByScore(this.allMergedRecords, this.leaderboardParams.orderingType);

        this.updateScoresState();
    };

    private onLeaderboardInfoRetrievedFailed = (status) => {
        this.printDebug("Failed to retrieve scores. Status = " + status);
        this.leaderboardVisualsControl.setFallbackEnabled(true);
    };

    private createLeaderboardRetrievalOption(leaderboardParameters: LeaderboardParams): Leaderboard.RetrievalOptions {
        const leaderboardRetrievalOption = Leaderboard.RetrievalOptions.create();
        leaderboardRetrievalOption.usersLimit = leaderboardParameters.userLimit;
        leaderboardRetrievalOption.usersType = leaderboardParameters.userType;

        return leaderboardRetrievalOption;
    }

    private validateCamera(): boolean {
        const camera: Camera = findParentComponent(this.getSceneObject(), 'Camera');
        if (isNull(camera) || camera.type !== Camera.Type.Orthographic) {
            return false;
        }
        return true;
    }

    private isInputValid(): boolean {
        if (this.leaderboardName === "") {
            this.printWarning("Please set leaderboard name. Important: changing leaderboard name will reset scores");
            return false;
        }

        if (isNull(this.getSceneObject().getParent()) || !this.validateCamera()) {
            this.printWarning("Please place Leaderboard under the Orthographic Camera");
            return false;
        }

        if (!isDateStringValid(this.leaderboardStartDate)) {
            this.printWarning("Please set correct date format, for example 31/12/2024");
            return false;
        }

        return true;
    }

    private reset() {
        this.currentUserRecord = null;
        this.allMergedRecords = [];
    }

    private recreateLeaderboard(): void {
        this.leaderboardParams.leaderboardDetails = this.createNewLeaderboardDetails(this.leaderboardParams.leaderboardDetails.leaderboardStartDate,
            this.leaderboardParams.leaderboardNamePrefix,
            this.leaderboardParams.ttlSeconds,
            this.leaderboardParams.scoreLifetimeOption,
            this.leaderboardParams.orderingType);

        this.leaderboardParams.leaderboardName = this.leaderboardParams.leaderboardDetails.leaderboardId;

        this.leaderboardVisualsControl.reset();
        this.createLeaderboard(this.leaderboardParams);
    }

    private onUpdate = () => {
        this.updateLeaderboardDetails();
        this.leaderboardVisualsControl.setTimerUiEnabled(this.useTimer);
    };
}
